"""
Build script to create portable Windows executables for the patent tools.
This script uses PyInstaller to create standalone .exe files.
"""

import os
import sys
import subprocess
import shutil

def check_and_install_requirements():
    """Check if required packages are installed and install if needed."""
    required_packages = {
        'pyinstaller': 'pyinstaller',
        'requests': 'requests',
        'beautifulsoup4': 'bs4'
    }
    
    print("Checking required packages...")
    
    for package_name, import_name in required_packages.items():
        try:
            __import__(import_name)
            print(f"✓ {package_name} is installed")
        except ImportError:
            print(f"✗ {package_name} not found. Installing...")
            subprocess.check_call([sys.executable, "-m", "pip", "install", package_name])
            print(f"✓ {package_name} installed successfully")

def create_spec_file(script_name, exe_name, icon_path=None):
    """Create a PyInstaller spec file for better control over the build."""
    
    icon_line = f"icon='{icon_path}'," if icon_path else ""
    
    spec_content = f"""
# -*- mode: python ; coding: utf-8 -*-

block_cipher = None

a = Analysis(
    ['{script_name}'],
    pathex=[],
    binaries=[],
    datas=[],
    hiddenimports=['requests', 'bs4', 'beautifulsoup4', 'urllib3', 'certifi', 'charset_normalizer', 'idna'],
    hookspath=[],
    hooksconfig={{}},
    runtime_hooks=[],
    excludes=[],
    win_no_prefer_redirects=False,
    win_private_assemblies=False,
    cipher=block_cipher,
    noarchive=False,
)
pyz = PYZ(a.pure, a.zipped_data, cipher=block_cipher)

exe = EXE(
    pyz,
    a.scripts,
    a.binaries,
    a.zipfiles,
    a.datas,
    [],
    name='{exe_name}',
    debug=False,
    bootloader_ignore_signals=False,
    strip=False,
    upx=True,
    upx_exclude=[],
    runtime_tmpdir=None,
    console=True,
    disable_windowed_traceback=False,
    argv_emulation=False,
    target_arch=None,
    codesign_identity=None,
    entitlements_file=None,
    {icon_line}
)
"""
    
    spec_filename = f"{exe_name}.spec"
    with open(spec_filename, 'w') as f:
        f.write(spec_content)
    
    return spec_filename

def build_executable(script_path, exe_name, description):
    """Build a single executable using PyInstaller."""
    
    print(f"\n{'='*60}")
    print(f"Building {exe_name}...")
    print(f"Description: {description}")
    print('='*60)
    
    # Check if script exists
    if not os.path.exists(script_path):
        print(f"Error: {script_path} not found!")
        return False
    
    # Create spec file for better control
    spec_file = create_spec_file(script_path, exe_name)
    
    # PyInstaller command
    cmd = [
        sys.executable, '-m', 'PyInstaller',
        '--clean',  # Clean PyInstaller cache and remove temporary files
        '--noconfirm',  # Replace output directory without asking
        spec_file
    ]
    
    try:
        # Run PyInstaller
        result = subprocess.run(cmd, capture_output=True, text=True)
        
        if result.returncode == 0:
            print(f"✓ {exe_name} built successfully!")
            
            # Move executable to main directory
            dist_exe = os.path.join('dist', f'{exe_name}.exe')
            if os.path.exists(dist_exe):
                shutil.move(dist_exe, f'{exe_name}.exe')
                print(f"✓ Moved {exe_name}.exe to current directory")
            
            return True
        else:
            print(f"✗ Error building {exe_name}")
            print("STDOUT:", result.stdout)
            print("STDERR:", result.stderr)
            return False
            
    except Exception as e:
        print(f"✗ Exception occurred: {str(e)}")
        return False

def cleanup():
    """Remove build artifacts."""
    print("\nCleaning up build artifacts...")
    
    dirs_to_remove = ['build', 'dist', '__pycache__']
    files_to_remove = ['*.spec']
    
    for dir_name in dirs_to_remove:
        if os.path.exists(dir_name):
            shutil.rmtree(dir_name)
            print(f"✓ Removed {dir_name}/")
    
    # Remove spec files
    import glob
    for pattern in files_to_remove:
        for file in glob.glob(pattern):
            os.remove(file)
            print(f"✓ Removed {file}")

def create_batch_files():
    """Create simple batch files to run the executables."""
    
    # Batch file for patent scraper
    scraper_bat = """@echo off
echo Starting Patent Scraper...
patent_scraper.exe
pause
"""
    
    # Batch file for patent comparer
    comparer_bat = """@echo off
echo Starting Patent Comparer...
patent_comparer.exe
pause
"""
    
    with open('run_scraper.bat', 'w') as f:
        f.write(scraper_bat)
    
    with open('run_comparer.bat', 'w') as f:
        f.write(comparer_bat)
    
    print("✓ Created batch files for easy execution")

def main():
    print("Patent Tools Executable Builder")
    print("==============================\n")
    
    # Show current directory and contents
    print(f"Current directory: {os.getcwd()}")
    print("\nFiles in current directory:")
    files = [f for f in os.listdir('.') if f.endswith('.py')]
    for f in files:
        print(f"  - {f}")
    print()
    
    # Check and install requirements
    check_and_install_requirements()
    
    # Try to find the script files with different possible names
    scraper_names = ['patent_scraper.py', 'patent-scraper.py', 'scraper.py']
    comparer_names = ['patent_comparer.py', 'patent-comparer.py', 'comparer.py']
    
    scraper_script = None
    comparer_script = None
    
    # Find scraper script
    for name in scraper_names:
        if os.path.exists(name):
            scraper_script = name
            print(f"Found scraper script: {name}")
            break
    
    # Find comparer script  
    for name in comparer_names:
        if os.path.exists(name):
            comparer_script = name
            print(f"Found comparer script: {name}")
            break
    
    if not scraper_script:
        print("\n⚠ Warning: Could not find patent scraper script!")
        print("  Expected one of:", ', '.join(scraper_names))
        print("  Make sure the file is in the current directory")
    
    if not comparer_script:
        print("\n⚠ Warning: Could not find patent comparer script!")
        print("  Expected one of:", ', '.join(comparer_names))
        print("  Make sure the file is in the current directory")
    
    print()
    
    # Build configurations
    builds = []
    
    if scraper_script:
        builds.append({
            'script': scraper_script,
            'exe_name': 'patent_scraper',
            'description': 'Google Patents Text Scraper - Extracts patent text to files'
        })
    
    if comparer_script:
        builds.append({
            'script': comparer_script,
            'exe_name': 'patent_comparer',
            'description': 'Patent Comparison Viewer - Creates side-by-side HTML comparisons'
        })
    
    if not builds:
        print("No scripts found to build!")
        input("\nPress Enter to exit...")
        return
    
    # Build each executable
    success_count = 0
    for build_config in builds:
        if build_executable(
            build_config['script'], 
            build_config['exe_name'], 
            build_config['description']
        ):
            success_count += 1
    
    # Clean up build artifacts
    cleanup()
    
    # Create batch files for easy execution
    if success_count > 0:
        create_batch_files()
    
    # Final summary
    print(f"\n{'='*60}")
    print("Build Summary:")
    print(f"{'='*60}")
    print(f"Successfully built: {success_count}/{len(builds)} executables")
    
    if success_count == len(builds):
        print("\n✓ All executables built successfully!")
        print("\nGenerated files:")
        if scraper_script:
            print("  - patent_scraper.exe    : Scrape patents from Google Patents")
            print("  - run_scraper.bat       : Double-click to run scraper")
        if comparer_script:
            print("  - patent_comparer.exe   : Compare patents side-by-side")
            print("  - run_comparer.bat      : Double-click to run comparer")
        print("\nThe .exe files are portable and can be copied to any Windows PC!")
    else:
        print("\n⚠ Some builds failed. Check the error messages above.")
    
    input("\nPress Enter to exit...")

if __name__ == "__main__":
    main()